//	Datatypes.h
//	Header file for the datatypes library.
//	These functions allow for easy translation of 
//	unknown data into known types.

#pragma once

#ifndef _DATATYPES_H_
#define _DATATYPES_H_

#include <OS.h>
#include <StorageDefs.h>
#include <InterfaceDefs.h>

struct Format;


class BBitmap;
class BView;
class DStream;
class BQuery;
class BMessage;


#include "DataFormats.h"

#pragma export on

typedef unsigned long DATAID;

struct DATAInfo {								//	Info about a specific translation
	ulong				formatType;				//	B_ASCII_TYPE, ...
	DATAID				formatHandler;			//	Internal to datatypes library
	ulong				formatGroup;			//	DATA_BITMAP, DATA_TEXT, ...
	float				formatQuality;			//	Quality of format in class 0.0-1.0
	float				formatCapability;		//	How much of the format do we do? 0.0-1.0
	char				formatName[B_FILE_NAME_LENGTH];
	char				MIMEName[B_FILE_NAME_LENGTH];
};




enum DATA_ERROR {
	DATA_BASE_ERROR		=	0xDADA0000,
	DATA_NO_HANDLER		=	DATA_BASE_ERROR,	//	no handler exists for data
	DATA_ILLEGAL_DATA,							//	data is not what it said it was
	DATA_NOT_INITIALIZED
};


#define DATA_CURRENT_VERSION 121
#define DATA_MIN_VERSION 110


extern "C" {

			//	you need to initialize the library to use it
			//	you should also shutdown it before quitting

	extern	const char *DATAVersion(	//	returns version string
							long & outCurVersion,	//	current version spoken
							long & outMinVersion);	//	minimum version understood

	extern	long		DATAInit(		//	establish connection
							ulong				mySignature,
							BQuery *			translators = NULL);

	extern	long		DATAShutdown();	//	don't want to talk anymore

			//	these functions call through to the translators
			//	when wantType is not 0, will only take into consideration 
			//	handlers that can read input data and produce output data

	extern	long		DATAIdentify(	//	find out what something is
							DStream &			inSource,
							BMessage *			ioExtension,
							DATAInfo &			outInfo,
							ulong				inHintType = 0,
							const char *		inHintMIME = NULL,
							ulong				inWantType = 0);

	extern	long		DATAGetHandlers(//	find all handlers for a type
							DStream &			inSource,
							BMessage *			ioExtension,
							DATAInfo * &		outInfo,	//	call delete[] on outInfo when done
							long &				outNumInfo,
							ulong				inHintType = 0,
							const char *		inHintMIME = NULL,
							ulong				inWantType = 0);

	extern	long		DATAGetAllHandlers(//	find all handler IDs
							DATAID * &			outList,//	call delete[] when done
							long &				outCount);

	extern long			DATAGetHandlerInfo(//	given a handler, get user-visible info
							DATAID				forHandler,
							const char * &		outName,
							const char * &		outInfo,
							long &				outVersion);

			//	note that handlers may choose to be "invisible" to
			//	the public formats, and just kick in when they 
			//	recognize a file format by its data.

	extern	long		DATAGetInputFormats(//	find all input formats for handler
							DATAID				forHandler,
							const Format * &	outFormats,//	don't write contents!
							long &				outNumFormats);

	extern	long		DATAGetOutputFormats(//	find all output formats for handler
							DATAID				forHandler,
							const Format * &	outFormats,//	don't write contents!
							long &				outNumFormats);

			//	actually do some work

	extern	long		DATATranslate(	//	morph data into form we want
							DStream &			inSource,
							const DATAInfo *	inInfo,//	may be NULL
							BMessage *			ioExtension,
							DStream &			outDestination,
							ulong 				inWantOutType,
							ulong				inHintType = 0,
							const char *		inHintMIME = NULL);

			//	For general viewing, each translator should support 
			//	creating a BView and returning an extent rectangle
			//	Note that the DStream will be given to the BView, and 
			//	should be deleted by the BView, clients should not 
			//	delete it after calling DATAMakeView.

	extern	long		DATAMakeView(
							DStream * 			inSource,
							const DATAInfo *	inInfo,//	may be NULL
							BMessage *			ioExtension,
							BView * &			outView,
							BRect &				outExtent,
							ulong				inHintType = 0,
							const char *		inHintMIME = NULL);


			//	Global settings for the datatypes library are 
			//	accessible through this interface

	extern	long		DATAGetSetting(
							const char *		inSettingName,
							void * &			outSetting,	//	call free() when done
							long &				outSize);

	extern	long		DATASetSetting(
							const char *		inSettingName,
							const void *		inSetting,
							long				inSize);

}

//	This setting name will return a list of record_refs for 
//	folders that contain translators. If you want to change 
//	this list at runtime and have the settings take effect, 
//	please close and reopen the datatypes library.
extern const char dataTranslatorFolders[];


#pragma export reset


#endif	// _DATATYPES_H_
